/* --------------------------------- oxplane.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Dynamics of the eXpermental plane.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"
#include "plane.h"

#define	GPLIMIT		(9*GACC)	/* +ve acceleration limit */
#define	GNLIMIT		(3*GACC)	/* -ve acceleration limit */
#define	GLIMIT		(10*GACC)	/* total G limit */

#define C_PI		3.1416

static int CCl = 0;

static void FAR
CCshow (OBJECT *p, int frac, char *title, long value)
{
	if (CC == p && (st.flags & SF_DEBUG)) {
		stroke_str  (1*8, CCl*8, title, 8, st.hfg);
		stroke_frac (6*8, CCl*8, value, 0, frac, 8, st.hfg);
		++CCl;
	}
}

static int FAR
check_land (OBJECT *p)
{
	int	errs;

	if (p->R[Z] > 0L)
		return (0);
	
	EX->flags |= PF_ONGROUND;

	errs = 0;

	if (!(EX->equip & EQ_GEAR)) {
		CCnote (p, "belly");
		++errs;
	}

	if (p->a[Y] > D90/9) {
		CCnote (p, "chinese");
		++errs;
	}
	p->a[Y] = 0;
	p->da[Y] = 0;
	if (p->a[X] < EP->gpitch) {
		CCnote (p, "nose down");
		++errs;
	}

	if (EX->v[X] < -10*VONE || EX->v[X] > 10*VONE) {
		CCnote (p, "sideways");
		++errs;
	}
	EX->v[X] = 0;

	if (p->V[Z] < -7*VONE) {
		CCnote (p, "crash");
		++errs;
	}
	p->V[Z] = 0;
	if (!on_runway (p)) {
		CCnote (p, "grass");
		++errs;
	}
	p->R[Z] = 0;
	if (!errs)
		CCland (p);
	return (errs);
}

static int FAR
check_takeoff (OBJECT *p)
{
	int	errs;

	errs = 0;
	if (!on_runway (p)) {
		CCnote (p, "grass");
		++errs;
	}
	if (!(EX->equip & EQ_GEAR)) {
		CCnote (p, "belly");
		++errs;
	}
	if (p->a[Y] > D90/9) {
		CCnote (p, "chinese");
		++errs;
	}
	if (p->a[X] < EP->gpitch)
		p->a[X] = EP->gpitch;
	if (!errs && p->R[Z] > 0L) {
		EX->flags &= ~PF_ONGROUND;
		CCfly (p);
	}
	return (errs);
}

/*
 * Formulaes used:
 *
 * V	airspeed
 * rho	air density
 * aoa	angle of attack
 * b	wing span
 * S	wing area
 *
 * aeff = aoa + flaps*flaps_effect + aoa0
 * Cl = aeff / Clrate
 * wing aspect ratio:
 *	AR = b^2 / S
 * k = 1/pi + ignored_small_factor
 * e = Oswald's efficiency factor
 * Cdi = k/e/AR * Cl^2
 * parasitic drag coefficient:
 * Cdp = Cdp_characteristic + Cd_flaps + Cd_gear + Cd_bombs + Cd_sppedbrakes
 * Cd = Cdi + Cdp
 * dynamic pressure:
 *	q = rho * V^2 / 2
 * lift is perpendicular to airflow:
 * 	lift = Cl * q * S
 * drag is parallel to airflow:
 * 	drag = Cd * q * S
 *
 * We need these factors:
 * Ixx	pitching moment of inertia
 * Iyy	rolling moment of inertia
 * Izz	yawing moment of inertia
 * we calculate the moments as:
 * Mx = lift*Lc + drag*Dc + thrust*Tc + (stablilizer+elevators)*Sc
 * Lc, Dc, Tc, Sc: effective moments levers.
 * My,Mz = f(many things...)
 *
*/

extern void FAR
dynamics_xplane (OBJECT *p, int interval)
{
	POINTER	*ptr;
	int	glimit, onground, taxiing;
	int	t, v, vmag, weight, thrust, push, drag, lift, k_ar;
	int	rho, sos, Cd, Cl, ClTail, Cdi, sa, ca, sb, cb;
	int	aFlaps, ClMax, ClMin, ClStall, Clrate;
	int	rVS, S, b2, b2VV, vmagV;
	int	Crudder, Cailerons, Celevators, Cvx, Cvz, Cdx, Cdy, Cdz;
	int	Fx, Fy, Fz, Mx, My, Mz;
	long	tt;
	ANGLE	alpha, beta, a;
	VECT	AA;
	AVECT	da;
CCl = 4;
	if (dynamics_input (p, interval))
		return;

	ptr = p->pointer;

	onground = EX->flags & PF_ONGROUND;
	glimit = 0;

	AA[X] = -fmul (GACC, p->T[X][Z]);	/* forces: gravity */
	AA[Y] = -fmul (GACC, p->T[Y][Z]);
	AA[Z] = -fmul (GACC, p->T[Z][Z]);

	EX->misc[5] = -AA[Z];			/* pilot's gravity */
	v = EX->v[Y];

	da[X] = da[Y] = da[Z] = 0;		/* moments */

	if (onground ? check_takeoff (p) : check_land (p)) {
		p->flags |= F_HIT;
		return;
	}

	taxiing = onground && p->speed < 40*VONE;	/* NWS */

	if (p->R[Z] < 14000L*VONE)
		rho = FONE - muldiv (FCON(.12), (int)(p->R[Z]/VONE), 3000);
	else
		rho = FCON(.4) - muldiv (FCON(.1),
					(int)(p->R[Z]/VONE)-14000, 3000);
	rho = fmul (rho, rho);		/* relative air density */
CCshow (p, 3, "rho", (long)fmul(rho, 1000));

	if (p->R[Z] <= 11000L*VONE)
		sos = 366*VONE + muldiv ((366-317)*VONE,
					(((int)p->R[Z])/VONE), 11000);
	else
		sos = 317*VONE;
CCshow (p, 0, "sos", (long)sos/VONE);
/*
 * Engine thrust diminishes linearly (roughly) with air density.
 * After-burner is relatively unaffected by altitude.
*/
	t = muldiv (EX->throttle, EP->mil_thrust, 100);
	t = fmul (rho, t);

	if (100 == EX->throttle)
		t += muldiv (EX->afterburner,
				EP->ab_thrust - EP->mil_thrust, 100);
	if (t < 0) {
		if (onground)
			t /= 2;		/* reverse thrust is 50% efficient? */
		else
			t = 0;
	}
	dampen (&EX->thrust, t, 8);
/*
 * sfc diminished to 0.8 (of its sea-level value) at 36000 feet then stays
 * stable.
*/
	t = EX->afterburner ? EP->ab_sfc : EP->mil_sfc;
	if (p->R[Z] < 11000L*VONE)
		t -= fmul (t, muldiv (FCON(0.2), (int)(p->R[Z]/VONE), 11000));
	else
		t -= fmul (t, FCON(0.2));
	t = muldiv (iabs(EX->thrust), t, 60*60/10);
	EX->misc[12] = t;
	EX->fuel -= TADJ(t);
	if (EX->fuel < 0) {
		EX->fuel = 0;
		EX->thrust = 0;
	}
/*
 * Automatically refuel when stationery on ground.
*/
	if (onground && 0 == p->speed && 0 == EX->thrust) {	/* refuel */
		if (EX->fuel < EP->fuel_capacity*100L) {
			EX->fuel += TADJ(500)*100L;		/* 500lb/sec */
			if (EX->fuel >= EP->fuel_capacity*100L)
				supply (p);
		}
	}

	thrust = fmul (EX->thrust*2, FCON(0.453*9.8/VONE*5));	/* N*VONE */
	tt = EP->weight + EX->fuel/100;				/* lb */
	weight = fmul ((int)(tt/VONE), FCON (0.453));		/* Kg*VONE */
	if ((t = EX->stores[WE_M61-1]) > 0)
		weight += (int)(t * st.bodies[O_M61]->shape->weight/1000/VONE);
	if ((t = EX->stores[WE_MK82-1]) > 0)
		weight += (int)(t * st.bodies[O_MK82]->shape->weight/1000/VONE);
	push = muldiv (thrust, VONE, weight);
CCshow (p, 0, "thrust", (long)thrust);
CCshow (p, 0, "weight", (long)weight);
CCshow (p, 0, "thrA", (long)push);

	if (taxiing) {
/*
 * Taxiing, ignore aerodynamics altogether. Only NWS and wheels friction.
*/
CCshow (p, 0, "vold", (long)p->speed);
		p->speed += TADJ(push);			/* v */
CCshow (p, 0, "vnew", (long)p->speed);
		drag = (EX->equip & EQ_GNDBRAKE) ? EP->BRAKE_MU : EP->WHEEL_MU;
		drag = TADJ (fmul (drag, GACC));	/* assume no lift */
CCshow (p, 0, "drag", (long)drag);
		if (p->speed > drag)
			p->speed -= drag;
		else if (p->speed < -drag)
			p->speed += drag;
		else
			p->speed = 0;			/* stop! */
CCshow (p, 0, "v", (long)p->speed);

		if (p->a[X] > EP->gpitch)
			p->a[X] -= TADJ(VD90);		/* slam down */
		if (p->a[X] < EP->gpitch)
			p->a[X] = EP->gpitch;

		if (p->a[Y])
			p->a[Y] -= TADJ(p->a[Y]);	/* level wings */

		EX->v[X] = 0;
		EX->v[Y] = fmul (p->speed, COS (-p->a[X]));
		EX->v[Z] = fmul (p->speed, SIN (-p->a[X]));

		p->da[X] = p->da[Y] = 0;
		p->da[Z] = muldiv (p->speed, EX->rudder, VONE*4); /* NWS */
		da[X] = da[Y] = da[Z] = 0;
	} else {
/*
 * Flying, although may still be on ground. Must do full arerodynamics but
 * allow for ground contact.
 *
 * Ideal Cl rate is 2*pi, and for a given aspect ratio A it is 2*pi*A/(A+2).
 * We represent angles as 2.0 angle = pi radians, so the formula is:
 * pi*pi*A/(A+2). We prefer to keep the inverse which is: 1/(pi*pi)*(1+2/A).
 * And to avoid fraction overflow inside the parens: 2/(pi*pi)*(0.5+1/A). We
 * keep 1/A in k_ar [which later is 1/(pi*A)].
*/
		k_ar = fdiv (EP->wing_area, EP->wing_span * EP->wing_span);
CCshow (p, 3, "1/AR", (long)fmul(k_ar, 1000));
		Clrate = fmul (FCON(2.0/(C_PI*C_PI)), FCON(0.5) + k_ar);
CCshow (p, 2, "Clrate", (long)ANG2DEG00 (Clrate));
		k_ar = fmul (k_ar, FCON(1.0/C_PI));		/* 1/(pi*AR) */
CCshow (p, 3, "k/AR", (long)fmul(k_ar, 1000));

		alpha = EX->v[Y] ? -ATAN (EX->v[Z], EX->v[Y]) : 0;
		alpha += EP->Aoffset;
		beta = p->speed ? ASIN (fdiv (EX->v[X], p->speed)) : 0;
CCshow (p, 2, "alpha", (long)ANG2DEG00 (alpha));
CCshow (p, 2, "beta", (long)ANG2DEG00 (beta));
		sa = SIN (alpha);
		ca = COS (alpha);
		sb = SIN (beta);
		cb = COS (beta);
		EX->misc[7] = alpha;

		a = alpha - EP->Cl0;
		ClTail = muldiv (EP->Tvol, a+EP->Toffset, Clrate);
CCshow (p, 3, "ClTail", (long)fmul(ClTail, 1000));

		aFlaps = fmul (EP->FEff, DEG2ANG (EX->flaps));
		a += aFlaps;
CCshow (p, 2, "aeff", (long)ANG2DEG00 (a));
/*
 * 'CLf' is to avoid overflow (fractions are limited to the range [-2...+2)).
*/
#define	CLf	4
		if (iabs(a/2) < (Uint)(CLf*Clrate))
			Cl = muldiv (FONE/CLf, a, Clrate);
		else if (a > 0)
			Cl = EP->MAXLIFT;
		else
			Cl = -EP->MAXLIFT;

		t = muldiv (FONE/CLf, aFlaps, Clrate);
		ClMax =  EP->MAXLIFT/CLf + t;
		ClMin = -EP->MAXLIFT/CLf + t;
		EX->wing_stall = 0;
		ClStall = FONE;
		if (Cl > ClMax) {
			if (EP->opt[4]) {
				Cl = Cl/4 - (Cl-ClMax);
				if (Cl < 0)
					Cl = 0;
				else
					Cl *= 4;
				EX->wing_stall = 1;
			} else {
				Cl = ClMax;
				ClStall = iabs (ca);
			}
		} else if (Cl < ClMin) {
			if (EP->opt[4]) {
				Cl = Cl/4 + (ClMin-Cl);
				if (Cl > 0)
					Cl = 0;
				else
					Cl *= 4;
				EX->wing_stall = 1;
			} else {
				Cl = ClMin;
				ClStall = iabs (ca);
			}
		}

		Cdi = fmul (Cl, k_ar*CLf);
		Cdi = fmul (Cdi, Cl)*CLf;
		Cdi = muldiv (Cdi, 100, EP->efficiency_factor);
		t = fmul (EP->wing_span*VONE, FCON(0.3048));
/*
 * The groung effect formula is extracted from the graph in Smiths 'The
 * Illustrated Guide To Aerodynamic' end of chapter 3.
*/
		if (p->R[Z] < (long)t) {		/* ground effect */
			t = fdiv ((int)p->R[Z], t);	/* h/b */
CCshow (p, 3, "h/b", (long)fmul(t, 1000));
			if (t < FCON(0.1))
				t = 5 * t;
			else
				t = FCON(1.06) - fdiv (FCON(0.07), t);
CCshow (p, 3, "gef", (long)fmul(t, 1000));
			Cdi = fmul (Cdi, t);
		}
CCshow (p, 3, "Cl", (long)fmul(Cl, CLf*1000));
CCshow (p, 3, "Cdi", (long)fmul(Cdi, 1000));

		Cd = EP->Cdp0;
CCshow (p, 3, "Cdp0", (long)fmul(Cd, 1000));
		Cd += Cdi;
CCshow (p, 3, "+Cdi", (long)fmul(Cd, 1000));
		if (EX->airbrake) {
			Cd += muldiv (EP->Cds, EX->airbrake, 100);
CCshow (p, 3, "+Brks", (long)fmul(Cd, 1000));
		}
		if (EX->equip & EQ_GEAR) {
			Cd += EP->Cdg;
CCshow (p, 3, "+Gear", (long)fmul(Cd, 1000));
		}
		if (EX->stores[WE_MK82-1] > 0) {
			Cd += EX->stores[WE_MK82-1] * EP->CdMK82;
CCshow (p, 3, "+MK82", (long)fmul(Cd, 1000));
		}
#define MYdz		Cl			/* is actualy 0.25*Cl */

		vmagV = p->speed;
		vmag = vmagV/VONE;
/*
 * 0.093 is for converting 'ft^2' to 'm^2'
*/
		S = fmul (EP->wing_area, FCON(0.093));
		rVS = muldiv (fmul (fmul (rho, FCON(1.225)), vmag), S, VONE);
CCshow (p, 0, "rVS", (long)rVS);
		b2 = fmul (EP->wing_span, FCON(0.3048/2));
CCshow (p, 0, "b2", (long)b2);
		b2VV = b2*VONE*VONE*4;

		Crudder = fdiv (EX->rudder, 57);
		Cailerons = muldiv (EP->MaxAilerons,
			EX->wing_stall ? rand()%81-40 : EX->ailerons,
			100);
		Celevators = muldiv (EP->MaxElevators,
			EX->wing_stall ? rand()%81-40 : EX->elevators,
			100);
		Cvx = EX->v[X];
		Cvz = EX->v[Z];
		Cdx = p->da[X];
		Cdy = p->da[Y];
		Cdz = -p->da[Z];
CCshow (p, 3, "Crdr", (long)fmul(Crudder, 1000));
CCshow (p, 3, "Cail", (long)fmul(Cailerons, 1000));
CCshow (p, 3, "Celev", (long)fmul(Celevators, 1000));
CCshow (p, 0, "Cvx", (long)Cvx);
CCshow (p, 0, "Cvz", (long)Cvz);
CCshow (p, 3, "Cdx", (long)fmul(Cdx, 1000));
CCshow (p, 3, "Cdy", (long)fmul(Cdy, 1000));
CCshow (p, 3, "Cdz", (long)fmul(Cdz, 1000));

		t = fmul (Cl*CLf, vmagV/2);
		lift = muldiv (rVS, t, weight);
CCshow (p, 0, "lift", (long)lift);
		t = fmul (Cd, vmagV/2);
		drag = muldiv (rVS, t, weight);
CCshow (p, 0, "drag", (long)drag);
/*
 * This formula is faking behaviour at high aoas (usually when stall is
 * disabled).
*/
#if 0
		drag += fmul (iabs(lift), FONE-ClStall);
#endif
		lift = fmul (lift, ClStall);
/*
 * Fx,Fy,Fz: aerodynamic forces on the plane, operating on the aero. center.
 * The results are the moments Mx,My,Mz about the CG.
*/
		t =	fmul (vmagV, fmul (EP->FXrudder, Crudder)) +
			fmul (Cvx, EP->FXvx)
			;
		Fx = muldiv (rVS, t, weight);
		Fx += fmul (sb, ihypot2d (drag, lift));
CCshow (p, 0, "Fx", (long)Fx);
		Fy = fmul (cb, fmul (sa, lift) - fmul (ca, drag));
CCshow (p, 0, "Fy", (long)Fy);
		Fz = fmul (cb, fmul (ca, lift) + fmul (sa, drag));
CCshow (p, 0, "Fz", (long)Fz);
/*
 * Examining the moment graphs at the end of chapter 13 of "The Design Of The
 * Aerplane" [Darrol Stinton, pub. Collins] one can roughly extract the
 * following values for weights around 15000Kg (the very top end):
 * Roll  moment (Ixx) = weight * 9.
 * Pitch moment (Iyy) = weight * 7.
 * yaw   moment (Izz) = weight * 17.
 * Note that our x,y,z are different!
*/
		t = fmul (ClTail, vmagV/2);
		t = muldiv (rVS, t, weight);	/* Tail lift */
		t = fmul (ca, t);
CCshow (p, 0, "Tlift", (long)t);

		t =	fmul (b2VV, fmul (EP->MXdx, Cdx)) +	/* damping */
			fmul (t, EP->MXtail) +		/* stabilizers */
			fmul (Fz, EP->MXlift) +		/* lift induced */
			fmul (Fy, EP->MXdrag) +		/* drag induced */
			fmul (push, EP->MXthrust) +	/* thrust induced */
			fmul (vmagV, fmul (EP->MXelevators, Celevators));
#if 0
#define MXelevators	FCON( 0.03) /* 0.05*/	/* elevators induced pitch */
#define MXdx		FCON(-0.02) /*-0.35*/	/* pitch damping */

#define Cm0	FCON( 0.04)
#define CmRate	FCON(-0.06)
		Cm = fmul (Cl-Cm0, CmRate);
		Cm = fmul (Cm, vmagV);
		t += Cm;
CCshow (p, 0, "Cm", (long)Cm);
#endif
		Mx = muldiv (rVS, b2*t, VONE*VONE*VONE);
CCshow (p, 0, "Mx", (long)Mx);

		t =	fmul (b2VV, fmul (EP->MYdy, Cdy)) +
			fmul (b2VV, fmul (MYdz, Cdz)) +
			fmul (EP->MYvx, Cvx) +
			fmul (vmagV, fmul (EP->MYailerons, Cailerons));
		My = muldiv (rVS, b2*t, VONE*VONE*VONE);
CCshow (p, 0, "My", (long)My);
		t =	fmul (b2VV, fmul (EP->MYdy, Cdy));
CCshow (p, 0, "t", (long)t);
		t = muldiv (rVS, b2*t, VONE*VONE*VONE);
CCshow (p, 0, "Mt", (long)t);
		t = muldiv (t, DEG2ANG(57)/9, weight);
CCshow (p, 0, "dayt", (long)t);

		t =	fmul (vmagV, fmul (EP->MZrudder, Crudder)) +
			fmul (b2VV, fmul (EP->MZdy, Cdy)) +
			fmul (b2VV, fmul (EP->MZdz, Cdz)) +
			fmul (EP->MZvx, Cvx) +
			fmul (vmagV, fmul (EP->MZailerons, Cailerons));
		Mz = muldiv (rVS, b2*t, VONE*VONE*VONE);
CCshow (p, 0, "Mz", (long)Mz);

		AA[X] += Fx;
		AA[Y] += Fy + push;
		AA[Z] += Fz;

		da[X] += muldiv (Mx, DEG2ANG(57)/7,       weight);
		da[Y] += muldiv (My, DEG2ANG(57)/9,       weight);
		da[Z] -= muldiv (Mz, DEG2ANG(57)/17*VONE, weight);
		if (EX->wing_stall) {
			da[X] += rand()%(VD90/16*2+1) - (VD90/16);
			da[Y] += rand()%(VD90/16*2+1) - (VD90/16);
			da[Z] -= rand()%(VD90/16*2+1) - (VD90/16);
		}

		if (onground) {
			t = p->speed/VONE - EP->liftoff_speed/2; /*nm->meter*/
			if (t < 0 && p->a[X] <= EP->gpitch)
				t = 0;
			if (da[X] > t)
				da[X] = t;
			if (da[X] < 0 && p->a[X] <= EP->gpitch)
				da[X] = 0;

			if (p->a[Y])
				p->a[Y] -= TADJ(p->a[Y]); /* level wings */

			da[Z] += -TADJ(p->da[Z]); /* no rotation */

			p->da[Y] = 0;
			da[Y] = 0;
/*
 * Effective weight is reduced by lift (it is assumed that the lift is
 * directed straight up which is reasonable when one is moving parallel to
 * the ground). We must be going fast enough for the brakes to be unable to
 * completely stop us within this interval.
*/
			drag = (EX->equip & EQ_GNDBRAKE)
				? EP->BRAKE_MU : EP->WHEEL_MU;
			drag = fmul (drag, lift - GACC);
			if (v < 0)
				drag = -drag;
			AA[Y] += fmul (ca, drag);
			AA[Z] += fmul (sa, drag);
			EX->wing_stall = 0;
		} else {
			EX->misc[5] += AA[Z];
			if (EX->misc[5] > GPLIMIT) {
				glimit = 1;
			} else if (EX->misc[5] < -GNLIMIT) {
				glimit = 1;
			}
			if (EX->equip & EQ_GEAR) {	/* gear shaking */
				t = ~1 & fmul (p->speed, FCON(0.02));
				t = t*t;
				AA[X] += TADJ(rand()%(1+t) - t/2);
				AA[Y] -= TADJ(rand()%(1+t));
				AA[Z] += TADJ(rand()%(1+t) - t/2);
			}
		}

		EX->v[X] += TADJ(AA[X]);
		EX->v[Y] += TADJ(AA[Y]);
		EX->v[Z] += TADJ(AA[Z]);
		VMmul (p->V, EX->v, p->T);
#if 0
		if (onground && p->V[Z] < 0) {
			p->V[Z] = 0;
			Mxpose (p->T);
			VMmul (EX->v, p->V, p->T);
			Mxpose (p->T);
		}
#endif
		p->speed = ihypot3d (p->V);
	}

	p->da[X] += TADJ (da[X])*VONE;	/* moments */
	p->da[Y] += TADJ (da[Y])*VONE;
	p->da[Z] += TADJ (da[Z]);

	da[X] = TADJ(p->da[X])*VONE;
	da[Y] = TADJ(p->da[Y])*VONE;
	da[Z] = TADJ(p->da[Z])*VONE;
	Myxz (p->T, da);

	if (!taxiing) {
		Mxpose (p->T);
		Vcopy (AA, EX->v);
		VMmul (EX->v, AA, p->T);
		Mxpose (p->T);
	}

	Mroty (p->T, p->a[Y]);
	Mrotx (p->T, p->a[X]);
	Mrotz (p->T, p->a[Z]);

	Mangles (p->T, p->a, da[Y]);

	if (onground && p->a[X] < EP->gpitch) {
		p->a[X] = EP->gpitch;
		Mobj (p);
	}

	if (taxiing) {
		VMmul (p->V, EX->v, p->T);
		p->V[Z] = 0;
	}

	if (onground && p->V[Z] < 0) {
		p->V[Z] = 0;
		Mxpose (p->T);
		VMmul (EX->v, p->V, p->T);
		EX->v[X] = 0;	/* testing */
		Mxpose (p->T);
		p->speed = ihypot3d (p->V);
	}

#define MAX_SPEED	1000
	if (p->speed > MAX_SPEED*VONE) {		/* temp */
		t = muldiv (FONE, (int)(MAX_SPEED*VONE), p->speed);
		EX->v[X] = fmul (t, EX->v[X]);
		EX->v[Y] = fmul (t, EX->v[Y]);
		EX->v[Z] = fmul (t, EX->v[Z]);
		p->V[X]  = fmul (t, p->V[X]);
		p->V[Y]  = fmul (t, p->V[Y]);
		p->V[Z]  = fmul (t, p->V[Z]);
		p->speed = ihypot3d (p->V);
	}

	EX->misc[10] = muldiv (p->speed, 1000, sos);

	if (p == CC) {
		st.indicators[0] = (p->R[Z] < 0) ? st.red : 0;
		st.indicators[1] = (p->R[Z]+p->V[Z] < 0) ? st.brown : 0;
		st.indicators[2] = (p->a[Y] > D90 || p->a[Y] < -D90)
					? st.green : 0;
		st.indicators[3] = EX->wing_stall ? st.red : 0;
		st.indicators[4] = glimit ? st.red : 0;
	}
}
