/* ------------------------------ joytest.c --------------------------------- */

/* An example of how to read PC joystick port by polling the hardware port
 * directly.
 * Uses inp()/outp() for byte port access.
 * Will timeout after JS_TIMEOUT reads. On my 486DX50, using my joystick port
 * and my joystick, full deflection reads 1600-1800 counts. I repeat the 'my'
 * above to indicate that each joystick card and joystick has different
 * characteristics. This shows as 6000-7000 counts if the READ_TIMER option
 * is used (depends on the mode the timer is set to...).
 *
 * This sample reads one port at a time. You can read both at once by
 * merging the two, but it will time out when either joystick is not connected.
 *
 * There is no need to optimize this routine since it runs for as long as
 * the joystick circuitry needs. Nevertheless, on slow machines increasing
 * the speed will yield higher resolution. This version is already optimized
 * for speed.
 *
 * About interrupts: these will cause some noise in the reading. The easiest
 * way around it is to read the stick twice and select the SMALLEST reading.
 * You may want to disable interrupts for the process (as the program shows
 * as comments) but watch out for lost serial-port characters and what not.
 * A middle way is to read the system timer which gets around a lot of the
 * noise; just #define USE_TIMER to enable this feature.
 * The enable/disable functions control the interrupts, you can use other
 * functions if your compiler provides them.
 *
 * usage:	>joytest		will read joystick 1
 *		>joytest x		will read joystick 2
 *
 * Written by Eyal Lebedinsky (eyal@ise.canberra.edu.au).
 * This version: April 1993.
*/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

extern int readjoy (int port, int *x, int *y, int *buttons);

#define USE_TIMER

#define JS_PORT		0x201
#define JS_TIMEOUT	5000
#define JS_READ		inp (JS_PORT)
#define WAITFORMASK	while (--i && !(~JS_READ & m))



#ifdef USE_TIMER
#define READING	get_timer ()

static short
disable (void)
{
	short	flags;

	_asm {
		pushf
		pop	flags
		cli
	}
	return (flags);
}

static void
enable (short flags)
{
	_asm {
		push	flags
		popf
	}
}

#define PIT_MODE	0x43		/* io port for command */
#define PIT_COUNTER_0	0x40		/* io port for timer 0 */

static unsigned int
get_timer (void)
{
	int	t;
	short	flags;

	flags = disable ();
	outp (PIT_MODE, 0);
	t = inp (PIT_COUNTER_0);
	t += inp (PIT_COUNTER_0) << 8;
	enable (flags);

	return (~t);
}
#else
#define READING	i
#endif



int						/* returns 0 if ok */
readjoy (int port, int *x, int *y, int *buttons)
{
	register int	i, m;
	int		timeout, mask;
#ifdef USE_TIMER
	unsigned int	t;
#else
/*	short		flags;*/
#endif

	i = JS_TIMEOUT;
	timeout = 0;

	mask = port ? 0x04 : 0x01;
	m = mask | (mask<<1);

#ifdef USE_TIMER
	t = READING;
#else
	/* flags = disable (); */
#endif
	outp (JS_PORT, 0);			/* trigger */
	WAITFORMASK;				/* wait for x or y */
	if (!i++) {
		timeout = 1;
		*x = *y = READING;
	} else if (JS_READ & mask) {	
		*y = READING;
		m = mask;
		WAITFORMASK;			/* wait for x */
		if (!i++)
			timeout = 1;
		*x = READING;
	} else {
		*x = READING;
		m = mask << 1;
		WAITFORMASK;			/* wait for y */
		if (!i++)
			timeout = 1;
		*y = READING;
	}
	i = JS_READ;				/* read buttons */

#ifndef USE_TIMER
	/* enable (flags); */
#endif

	m = mask << 4;
	*buttons = !(i & m) | ((!(i & (m<<1))) << 1);
#ifdef USE_TIMER
	*x -= t;
	*y -= t;
#else
	*x = JS_TIMEOUT - *x;
	*y = JS_TIMEOUT - *y;
#endif
	return (timeout);
}



/* This main() is for demonstration.
*/

int
main (int argc, char *argv[])
{
	int	rc, x, y, buttons;

	printf ("Hit any key to exit\n");
	while (!kbhit ()) {
		rc = readjoy (argc-1, &x, &y, &buttons);
		printf ("\rstatus=%x x=%5u y=%5u buttons=%x ",
			rc, x, y, buttons);
	}
	exit (0);
	return (0);
}
