/* --------------------------------- extern.h ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* mainly function prototypes.
*/

#ifndef REFRESH
#define	REFRESH	60	/* 60ms, aiming at 67=4@60Hz or 69=5@72Hz */
#endif

#define	BULLETV		(1000*VONE)
#define	BULLETSCATTER	(D90/180)

#define ALLOCATE(type)	(type *)xcalloc (sizeof (type), 1)

#define iabs(i)		((Uint)abs(i))

/* fly8.c (this is where main() is) */
extern struct status	st;

/* body.c */
extern BODY *	FAR new_body (void);
extern int	FAR bodies_init (void);
extern void	FAR bodies_term (void);
extern ONAME	FAR body_name (char *title);

/* cockpit.c */
extern void	FAR show_inst (VIEW *view, OBJECT *obj);

/* colors.c */
extern long	FAR hi_color (long color);
extern int	FAR menu_colors (void);

/* command.c */
extern int	FAR user_command (void);

/* device.c */
extern int	FAR GrPrintf (const char *fmt, ...);
extern int	FAR GrNote (const char *fmt, ...);
extern DEVICE *	FAR devices_select (char *dname);
extern struct GrDriver * FAR devices_init (char *name);
extern void	FAR devices_term (void);

/* getstr.c */
extern void	FAR show_prompt (VIEW *view, int orgx, int orgy, int maxx,
	int maxy, int bss);
extern void	FAR getstr (char *prompt, char FAR *str, int len);

/* hud.c */
extern void     FAR hud_setup (OBJECT *p);

/* ifuncs.c */
#define	D180		0x8000U
#define	D90		0x4000
#define	VD90		(D90/VONE)
#define	VD180		(VD90*2)
#define	VD360		(VD90*4)
#define	ANG2DEG(x)	muldiv ((x), 90, D90)
#define	ANG2DEG00(x)	muldiv ((x), 9000, D90)
#define	DEG2ANG(x)	muldiv ((x), D90, 90)
#define	SIN(a)		my_sin (a)
#define	COS(a)		my_sin ((a)+D90)
#define	ASIN(i)		my_asin(i)		/* [-90...+90] */
#define	ACOS(i)		(D90 - my_asin (i))	/* [0...180] */
						/* note: 180 is -ve! */
#define	ATAN(iy,ix)	my_atan((iy),(ix))
#define	SQRT(ul)	my_sqrt(ul)

extern Uint	FAR FASTCALL est_hyp (int x, int y, int z);
extern Uint	FAR FASTCALL est_dist (LVECT R1, LVECT R2);
extern Uint	FAR FASTCALL ihypot2d (int x, int y);
extern Uint	FAR FASTCALL ihypot3d (VECT A);
extern Ulong	FAR FASTCALL lhypot3d (LVECT A);
extern Ulong	FAR FASTCALL ldist3d (LVECT A, LVECT B);

extern int	FAR init_funcs (void);
extern void	FAR term_funcs (void);
extern int	FAR FASTCALL my_sin (ANGLE angle);
extern ANGLE	FAR FASTCALL my_asin (int sin);
extern ANGLE	FAR FASTCALL my_atan (int y, int x);
extern Uint	FAR FASTCALL my_sqrt (Ulong x);
extern int	FAR FASTCALL lin2log (int linear);
extern void	FAR dampen (short *old, int new, int factor);

/* info.c */
extern Uchar *	FAR show_time (char *title, Ulong tt);
extern void FAR screen_info (VIEW *view, int orgx, int orgy, int maxx,
	int maxy, int bss, int mode);

/* init.c */
extern void     FAR welcome (void);
extern void	FAR initialize (char *argv[]);

/* keyboard.c */
extern struct KbdDriver * FAR keyboard_init (char *name);
extern void	FAR keyboard_term (void);

/* land.c */
extern int	FAR land_init (void);
extern void	FAR land_term (void);
extern int	FAR land_update (OBJECT *pov);

/* log.c */
extern int	FAR log_init (void);
extern void	FAR log_term (void);
extern int	FAR LogPrintf (const char *fmt, ...);

/* loop.c */
extern void	FAR idle_loop (void);
extern void	FAR buffers_free (void);
extern void     FAR double_buffer (int mode);
extern void	FAR reset_page (int empty);
extern void	FAR show_fixed (int del);
extern BUFFER * FAR new_buffer (void);
#if 1
extern int	FAR FASTCALL add_line (int x, int y, int t);
#else
#define add_line(xx,yy,tt) \
	This is not done yet!
#endif
extern int	FAR add_5op (int t, int a, int b, int c, int d, int e);
extern OBJECT *	FAR get_viewer (int type);
extern void	FAR save_viewport (OBJECT *p);
extern void	FAR get_viewport (OBJECT *p);

/* macros.c */
extern int	FAR mread (void);
extern int	FAR mgetch (void);
extern int	FAR mac_interpret (int *keys, int len);
extern void	FAR mac_flush (void);
extern int	FAR mac_init (void);
extern void	FAR mac_term (void);

/* mat.c */
#define	Mcopy(to,from)	memcpy (to, from, sizeof (MAT))
#define	Vcopy(to,from)	memcpy (to, from, sizeof (VECT))
#define	LVcopy(to,from)	memcpy (to, from, sizeof (LVECT))
#define	AVcopy(to,from)	memcpy (to, from, sizeof (AVECT))
#define	Vinc(to,from)	((to)[X]+=(from)[X], \
			 (to)[Y]+=(from)[Y], \
			 (to)[Z]+=(from)[Z])
#define	Vdec(to,from)	((to)[X]-=(from)[X], \
			 (to)[Y]-=(from)[Y], \
			 (to)[Z]-=(from)[Z])
#define	Vadd(to,a,b)	((to)[X]=(a)[X]+(b)[X], \
			 (to)[Y]=(a)[Y]+(b)[Y], \
			 (to)[Z]=(a)[Z]+(b)[Z])
#define	Vsub(to,a,b)	((to)[X]=(a)[X]-(b)[X], \
			 (to)[Y]=(a)[Y]-(b)[Y], \
			 (to)[Z]=(a)[Z]-(b)[Z])

extern void	FAR Mident (MAT m);
extern void	FAR Mrotx (MAT m, ANGLE d);
extern void	FAR Mroty (MAT m, ANGLE d);
extern void	FAR Mrotz (MAT m, ANGLE d);
extern void	FAR Mxpose (MAT m);
extern void	FAR Mmul (MAT m, MAT t);
extern void	FAR Mangles (MAT m, AVECT a, ANGLE dy);
extern void	FAR Mangles1 (MAT m, AVECT a);		/* obsolete */
extern void	FAR Mobj (OBJECT *p);
extern void	FAR Myxz (MAT m, AVECT a);
extern void	FAR Vscale (VECT a, VECT b, int scalar);
extern void	FAR Vmuldiv (VECT a, VECT b, int m, int d);

/* memory.c */
extern void *	FAR xcalloc (Uint size, Uint count);
extern void *	FAR xmalloc (Uint size);
extern void *	FAR xfree (void *block);
extern void	FAR memory_check (void);
extern void *	FAR memory_alloc (Uint bytes);
extern void *	FAR memory_free (void *block, int bytes);
extern int	FAR memory_init (void);
extern void	FAR memory_term (void);
extern int	FAR memory_get (int bytes, int count);
#if 0
extern void *	FAR list_del (void *member, void **head, void **tail);
extern void	FAR fifo_put (void *head, void *new);
extern void *	FAR fifo_get (void *head);
#endif

#define	NEW(p)	((p) = memory_alloc (sizeof (*(p))))
#define	DEL(p)	memory_free ((p), sizeof (*(p)))
#define	DEL0(p)	(memory_free ((p), sizeof (*(p))), (p) = 0)

/* menu.c */
typedef struct menu	MENU;
struct menu {
	int	letter;
	char	*text;
};

#define	MENU_ABORTED	-1
#define	MENU_FAILED	-2

extern char	FAR menuch[];
extern int	FAR menu_open (MENU *menu, int selected);
extern void	FAR menu_close (void);
extern int	FAR hdd_menu (VIEW *view, int orgx, int orgy, int maxx,
	int maxy);
extern void	FAR show_menu (VIEW *view, int orgx, int orgy, int maxx,
	int maxy, int bss);

/* menuhud.c */
extern int	FAR menu_hudtype (void);
extern int	FAR menu_hud (void);

/* menus.c */
extern void	FAR emit_drone (void);
extern void	FAR set_small_frame (void);
extern int	FAR menu_top (void);
extern void	FAR set_screen (int sizex, int sizey);

/* message.c */
extern void	FAR msg_del (const HMSG *p);
extern HMSG *	FAR MsgPrintf (int ttl, const char *fmt, ...);
extern HMSG *	FAR MsgEPrintf (int ttl, const char *fmt, ...);
extern HMSG *	FAR MsgWPrintf (int ttl, const char *fmt, ...);
extern void	FAR msg_show (int orgx, int orgy, int maxx, int maxy, int bss);
extern void	FAR msg_clear (void);
extern int	FAR msg_init (void);
extern void	FAR msg_term (void);

/* netport.c */
extern int	FAR netports_init (void);
extern void	FAR netports_term (void);
extern void	FAR netports_receive (void);
extern void	FAR netport_count (PLAYER *pl, int delta);
extern int	FAR packet_deliver (PACKET *pack);
extern int	FAR packet_send (PACKET *pack, int mode);
extern PACKET *	FAR packet_new (Ushort size);
extern PACKET *	FAR packet_del (PACKET *pack);
extern void	FAR PlName (PLAYER *pl);

/* object.c */
extern OBJECT *	FAR create_object (ONAME name, int init);
extern OBJECT *	FAR create_land (ONAME name);
extern OBJECT *	FAR delete_object (OBJECT *object);
extern OBJECT *	FAR delete_land (OBJECT *object);
extern void	FAR list_clear (OBJECT *list[]);
extern void	FAR shape_free (VERTEX *vx);
extern int	FAR shape_read (SHAPE *shape, char *VxFileName);

/* objects.c */
extern void	FAR object_hit (OBJECT *obj, int seed, int speed, int extent,
	int damaging);
extern int	FAR body_init (BODY *b);
extern void	FAR body_term (BODY *b);
extern void	FAR body_delete (OBJECT *p);
extern void	FAR body_dynamics (OBJECT *p, int interval);
extern void	FAR body_hit (OBJECT *p, int speed, int extent, int damaging);
extern int	FAR objects_dynamics (int interval);

/* oplane.c */
extern void	FAR eject (OBJECT *obj);

/*pid.c */
extern long	FAR pid_calc (PID *pid, long P, int interval);

/* player.c */
extern int	FAR players_init (void);
extern void	FAR players_term (void);
extern void	FAR players_delete (void);
extern void	FAR players_remove (void);
extern void	FAR players_purge (void);
extern void	FAR players_flush (void);

extern PLAYER *	FAR player_add (PACKET *pack);
extern PLAYER *	FAR player_delete (PLAYER *player);
extern PLAYER *	FAR player_find (PACKET *pack);
extern PLAYER *	FAR player_next (PLAYER *player);
extern PLAYER *	FAR player_active (PACKET *pack);
extern void	FAR player_remove (PLAYER *player);

/* pointer.c */
extern int	FAR pointers_init (void);
extern void	FAR pointers_term (void);
extern POINTER * FAR pointer_select (char *pname);
extern POINTER * FAR pointer_release (POINTER *ptr);
extern void	FAR std_key (POINTER *p, int key);
extern int      FAR menu_ptrs (void);

/* remote.c */

#define	RC_SENDOK	0x00
#define	RC_PACKED	0xfe
#define	RC_SENDFAIL	0xff

extern int	FAR remote_urhit (OBJECT *p, int speed, int extent,
	int damaging);
extern int	FAR remote_imhit (OBJECT *p, int seed, int speed, int extent,
	int damaging);
extern void	FAR shoot (OBJECT *p, int weapon, int n, int seed, int interval);
extern int	FAR crc (PACKET *pack);
extern int	FAR remote_init (void);
extern void	FAR remote_term (void);
extern int	FAR send_obj (OBJECT *p, PLAYER *player, int mode);
extern void	FAR remote_receive (OBJECT *p);
extern void	FAR remote_ping (void);
extern void	FAR remote_play (PLAYER *pl);
extern void	FAR remote_noplay (PLAYER *pl, int mode);
extern void	FAR remote_reply (PLAYER *pl, int reply);
extern void	FAR remote_shoot (OBJECT *p, int weapon, int n, int seed,
	int interval);
extern void	FAR remote_msg (char *text, PLAYER *pl, int mode);
extern void	FAR remote_time (PLAYER *pl, int mode);
extern void	FAR remote_refresh (void);

/* show.c */
extern void	FAR objects_show (VIEW *view, OBJECT *pov, int color,
	LVECT OR, VECT RR);

/* sky.c */
extern int	FAR sky_init (void);
extern void	FAR sky_term (void);
extern void	FAR show_sky (VIEW *view, OBJECT *p, int color, int shift);

/* sound.c */
extern struct SndDriver * FAR sound_init (char *name);
extern void	FAR sound_term (void);

/* stroke.c */
extern void	FAR stroke_angle (ANGLE angle);
extern int	FAR num_size (long num, int ss);
extern void	FAR num_extent (long num, int ss, int *exs, int *exc,
	int *eys, int *eyc);
extern int	FAR stroke_size (char *s, int size);
extern int	FAR stroke_char (int x, int y, int digit, int size, int color);
extern int	FAR stroke_str (int x, int y, char *p, int size, int color);
extern int	FAR stroke_num (int x, int y, long num, int size, int color);
extern int	FAR stroke_frac (int x, int y, long num, int digits, int frac,
	int size, int color);

/* system.c */
extern void	FAR sys_poll (void);

/* term.c */
extern void	FAR terminate (int StackUsed);
extern void	FAR die (void);

/* tunes.c */
extern int	FAR TnHello[];
extern int	FAR TnGone[];
extern int	FAR TnHit[];
extern int	FAR TnEngine[];
extern int	FAR TnAlarm[];
extern int	FAR TnWarn[];
extern int	FAR TnNotice[];
extern int	FAR TnGear[];
extern int	FAR TnLand[];
extern int	FAR TnFly[];
extern int	FAR TnDamage[];
extern int	FAR TnTune4[];

/* util.c */
extern int	FAR opt36 (int c);
extern char *	FAR get_arg (char *options, char *argname);
extern long	FAR get_narg (char *options, char *argname);
extern char *	FAR get_iarg (char *options, int argno);
extern long	FAR get_inarg (char *options, int argno);
extern char *	FAR get_parg (char *, int);
extern void	FAR randomize (void);

/* views.c */
extern void	FAR show_main (VIEW *view, OBJECT *p, int shift, int mode);
extern int	FAR scenery (int type);
extern void	FAR show_hdd (void);
extern int	FAR menu_view (int type);

/* vmodes.c */
extern int	FAR vm_read (void);
extern void	FAR vm_free (void);

/* window.c */
extern void	FAR windows_term (void);
extern void	FAR windows_set (void);
extern void	FAR set_screen (int sizex, int sizey);
extern void	FAR adjust_viewports (void);
extern void	FAR set_viewport (void);
extern void	FAR set_textwin (void);
extern int	FAR menu_windows (void);
extern void	FAR zoom (VIEWPORT *vp, int zoom);
extern void	FAR clear_text (void);

/*
 * Fixed point math. 'n' is number of fraction bits.
*/

#define	FnMUL(n,x,y)	((int)(((x) * (long)(y)) >> (n)))
#define	FnDIV(n,x,y)	((int)(((long)(x) << (n)) / (y)))
#define	FnCON(n,c)	((int)((1 << (n)) * (double)(c)))
#define	FnONE(n)	FnCON(n,1)

#define FSCALE		14		/* fraction bits in sine/cos etc. */
#define	FCON(c)		FnCON (FSCALE, (c))
#define	FONE		FnONE(FSCALE)
#define	fdiv(x,y)	FnDIV (FSCALE, (x), (y))
#define	fuscale(x)	((x) >> FSCALE)

#define VSCALE		4		/* fraction bits in v (speed) */
#define	VONE		FnONE(VSCALE)
#define	VMAX		0x7fff
#define	vscale(x)	((int)((x) * (long)VONE))
#define	vuscale(x)	((x) >> VSCALE)

#ifdef	DEBUG_MULDIV
#ifdef	USE_ASM
#undef	USE_ASM
#endif
#endif

#ifdef	USE_ASM

extern int	FAR FASTCALL fmula (int x, int y);
#define	fmul	fmula

extern int	FAR FASTCALL muldiva (int x, int y, int z);
#define	muldiv	muldiva

extern int	FAR FASTCALL dithadja (int x, int dither, int interval);
#define	dithadj	dithadja

extern void FAR Vmula (VECT R, VECT V, MAT M);
#define	VMmul	Vmula

extern void FAR matyxz (MAT T, int spitch, int cpitch, int sroll, int croll,
	int syaw, int cyaw);
#define	build_mat matyxz

#define	LL(x)	((int)(x))

#else	/* ifdef USE_ASM */

#define dithadj(x,y,z)	((int)(((x) * (long)(z) + ((x)>=0?(y):-(y))) / 1000))

#ifdef	DEBUG_MULDIV

#define	fmul(x,y)	fmulchk ((x), (y), __FILE__, __LINE__)
extern int	FAR fmulchk (long x, long y, char *file, int line);

#define	muldiv(x,y,z)	muldivchk ((x), (y), (z), __FILE__, __LINE__)
extern int	FAR muldivchk (long x, long y, long z, char *file, int line);

#define LL(x)	lcheck ((x), "LL", __FILE__, __LINE__)
extern long	FAR lcheck (long x, char *name, char *file, int line);

#else	/* ifdef DEBUG_MULDIV */

#define muldiv(x,y,z)	((int)(((x) * (long)(y)) / (z)))
#define	fmul(x,y)	((int)(((x) * (long)(y)) >> FSCALE))

#endif	/* ifdef DEBUG_MULDIV */

#define VMmul(R,V,M) \
	(R[X]=fmul(V[X],M[X][X])+fmul(V[Y],M[Y][X])+fmul(V[Z],M[Z][X]), \
	 R[Y]=fmul(V[X],M[X][Y])+fmul(V[Y],M[Y][Y])+fmul(V[Z],M[Z][Y]), \
	 R[Z]=fmul(V[X],M[X][Z])+fmul(V[Y],M[Y][Z])+fmul(V[Z],M[Z][Z]))

extern void	FAR cbuild_matyxz (MAT T, int spitch, int cpitch, int sroll,
	int croll, int syaw, int cyaw);
#define build_mat cbuild_matyxz

#endif	/* ifdef USE_ASM */

#define	TADJ(x)		dithadj ((x), st.dither, st.interval)
